<?php

/**
 * Class Differentiator
 *
 * Maneja todas las operaciones de la base de datos para la tabla 'differentiators'.
 */
class Differentiator
{
    private PDO $db;
    private string $table_name = "differentiators";

    // Propiedades del objeto Differentiator
    public ?int $id = null;
    public ?string $title = null;
    public ?string $description = null;
    public ?string $icon = null;
    public ?int $display_order = 0;

    public function __construct(PDO $db)
    {
        $this->db = $db;
    }

    /**
     * Lee todos los registros, ordenados por 'display_order'.
     * @return PDOStatement
     */
    public function readAll(): PDOStatement
    {
        $query = "SELECT * FROM " . $this->table_name . " ORDER BY display_order ASC, id ASC";
        $stmt = $this->db->prepare($query);
        $stmt->execute();
        return $stmt;
    }

    /**
     * Lee un único registro por su ID y carga los datos en el objeto.
     * @return bool True si se encuentra, false si no.
     */
    public function readOne(): bool
    {
        $query = "SELECT * FROM " . $this->table_name . " WHERE id = :id LIMIT 1";
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(':id', $this->id);
        $stmt->execute();
        $row = $stmt->fetch(PDO::FETCH_ASSOC);

        if ($row) {
            $this->title = $row['title'];
            $this->description = $row['description'];
            $this->icon = $row['icon'];
            $this->display_order = (int)$row['display_order'];
            return true;
        }
        return false;
    }

    /**
     * Crea un nuevo registro en la base de datos.
     * @return bool True si se crea exitosamente, false si no.
     */
    public function create(): bool
    {
        $query = "INSERT INTO " . $this->table_name . " (title, description, icon, display_order) VALUES (:title, :description, :icon, :display_order)";
        $stmt = $this->db->prepare($query);

        // Limpiar datos
        $this->title = htmlspecialchars(strip_tags($this->title));
        $this->description = htmlspecialchars(strip_tags($this->description));
        $this->icon = htmlspecialchars(strip_tags($this->icon));

        // Vincular parámetros
        $stmt->bindParam(':title', $this->title);
        $stmt->bindParam(':description', $this->description);
        $stmt->bindParam(':icon', $this->icon);
        $stmt->bindParam(':display_order', $this->display_order, PDO::PARAM_INT);

        return $stmt->execute();
    }

    /**
     * Actualiza un registro existente en la base de datos.
     * @return bool True si se actualiza exitosamente, false si no.
     */
    public function update(): bool
    {
        $query = "UPDATE " . $this->table_name . " 
                  SET title = :title, description = :description, icon = :icon, display_order = :display_order
                  WHERE id = :id";
        $stmt = $this->db->prepare($query);

        // Limpiar datos
        $this->title = htmlspecialchars(strip_tags($this->title));
        $this->description = htmlspecialchars(strip_tags($this->description));
        $this->icon = htmlspecialchars(strip_tags($this->icon));

        // Vincular parámetros
        $stmt->bindParam(':id', $this->id, PDO::PARAM_INT);
        $stmt->bindParam(':title', $this->title);
        $stmt->bindParam(':description', $this->description);
        $stmt->bindParam(':icon', $this->icon);
        $stmt->bindParam(':display_order', $this->display_order, PDO::PARAM_INT);

        return $stmt->execute();
    }

    /**
     * Elimina un registro de la base de datos por su ID.
     * @return bool True si se elimina exitosamente, false si no.
     */
    public function delete(): bool
    {
        $query = "DELETE FROM " . $this->table_name . " WHERE id = :id";
        $stmt = $this->db->prepare($query);
        $stmt->bindParam(':id', $this->id, PDO::PARAM_INT);
        return $stmt->execute();
    }
}