<?php

require_once ROOT_PATH . '/app/models/Testimonial.php';
require_once ROOT_PATH . '/app/models/Project.php'; // Para el selector de proyectos

/**
 * Class TestimonialController
 *
 * Gestiona la lógica CRUD para los testimonios, incluyendo la subida de avatares.
 */
class TestimonialController
{
    private PDO $db;
    private Testimonial $testimonial;
    private Project $project;

    public function __construct(PDO $db)
    {
        $this->db = $db;
        $this->testimonial = new Testimonial($this->db);
        $this->project = new Project($this->db);
    }

    /**
     * Muestra la lista de todos los testimonios.
     */
    public function index(): void
    {
        if (!isset($_SESSION['user_id'])) {
            redirect('login');
        }

        $stmt = $this->testimonial->readAll();
        $testimonials = $stmt->fetchAll(PDO::FETCH_ASSOC);

        ob_start();
        require_once ROOT_PATH . '/app/views/admin/testimonials/index.php';
        $content = ob_get_clean();
        $this->renderAdminLayout($content, "Gestionar Testimonios");
    }

    /**
     * Muestra el formulario para crear un nuevo testimonio o procesa su creación.
     */
    public function create(): void
    {
        if (!isset($_SESSION['user_id'])) {
            redirect('login');
        }

        $projects = $this->project->readAll()->fetchAll(PDO::FETCH_ASSOC);
        $error = null;
        $testimonialData = [];

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $testimonialData = $_POST;
            
            // 1. Manejar la subida de la imagen del avatar
            $uploadResult = $this->handleImageUpload('avatar');

            if ($uploadResult['success']) {
                // 2. Asignar datos al objeto Testimonial
                $this->testimonial->name = $_POST['name'] ?? '';
                $this->testimonial->role = $_POST['role'] ?? '';
                $this->testimonial->company = $_POST['company'] ?? '';
                $this->testimonial->quote = $_POST['quote'] ?? '';
                $this->testimonial->related_project_id = !empty($_POST['related_project_id']) ? $_POST['related_project_id'] : null;
                $this->testimonial->avatar = $uploadResult['path']; // Guardar la ruta de la nueva imagen

                // 3. Intentar crear en la base de datos
                if ($this->testimonial->create()) {
                    $_SESSION['message'] = "Testimonio creado exitosamente.";
                    redirect('admin/testimonials');
                } else {
                    $error = "Falló al crear el testimonio.";
                    // Si falla la DB, borrar la imagen que acabamos de subir para no dejar archivos huérfanos
                    if ($uploadResult['path'] && file_exists(ROOT_PATH . '/public' . $uploadResult['path'])) {
                        @unlink(ROOT_PATH . '/public' . $uploadResult['path']);
                    }
                }
            } else {
                // Si la subida de imagen falló, mostrar el error
                $error = $uploadResult['error'];
            }
        }

        ob_start();
        require_once ROOT_PATH . '/app/views/admin/testimonials/create.php';
        $content = ob_get_clean();
        $this->renderAdminLayout($content, "Crear Nuevo Testimonio");
    }

    /**
     * Muestra el formulario para editar un testimonio o procesa la actualización.
     */
    public function edit(): void
    {
        if (!isset($_SESSION['user_id'])) {
            redirect('login');
        }
        
        $this->testimonial->id = $_GET['id'] ?? 0;
        if (!$this->testimonial->readOne()) {
            $_SESSION['error'] = "Testimonio no encontrado.";
            redirect('admin/testimonials');
        }

        $projects = $this->project->readAll()->fetchAll(PDO::FETCH_ASSOC);
        $error = null;
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $currentAvatar = $this->testimonial->avatar;
            
            // Opción para eliminar la imagen existente
            if (isset($_POST['delete_avatar'])) {
                if ($currentAvatar && file_exists(ROOT_PATH . '/public' . $currentAvatar)) {
                    @unlink(ROOT_PATH . '/public' . $currentAvatar);
                }
                $this->testimonial->avatar = null;
            }
            
            // Manejar la subida de una nueva imagen (reemplazará la anterior si existe)
            $uploadResult = $this->handleImageUpload('avatar', $currentAvatar);
            
            if ($uploadResult['success']) {
                $this->testimonial->name = $_POST['name'] ?? '';
                $this->testimonial->role = $_POST['role'] ?? '';
                $this->testimonial->company = $_POST['company'] ?? '';
                $this->testimonial->quote = $_POST['quote'] ?? '';
                $this->testimonial->related_project_id = !empty($_POST['related_project_id']) ? $_POST['related_project_id'] : null;
                
                // Solo actualiza la ruta de la imagen si se subió una nueva
                if ($uploadResult['path'] !== null) {
                    $this->testimonial->avatar = $uploadResult['path'];
                }

                if ($this->testimonial->update()) {
                    $_SESSION['message'] = "Testimonio actualizado exitosamente.";
                    redirect('admin/testimonials');
                } else {
                    $error = "Falló al actualizar el testimonio.";
                }
            } else {
                $error = $uploadResult['error'];
            }
        }

        ob_start();
        require_once ROOT_PATH . '/app/views/admin/testimonials/edit.php';
        $content = ob_get_clean();
        $this->renderAdminLayout($content, "Editar Testimonio");
    }

    /**
     * Procesa la eliminación de un testimonio y su avatar asociado.
     */
    public function delete(): void
    {
        if (!isset($_SESSION['user_id'])) {
            redirect('login');
        }
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $this->testimonial->id = $_POST['id'] ?? 0;

            // Primero, obtenemos los datos del testimonio para saber qué imagen borrar
            if ($this->testimonial->readOne()) {
                $avatarToDelete = $this->testimonial->avatar;

                // Ahora, intentamos borrar de la base de datos
                if ($this->testimonial->delete()) {
                    // Si se borró de la DB, borramos el archivo de imagen físico
                    if ($avatarToDelete && file_exists(ROOT_PATH . '/public' . $avatarToDelete)) {
                        @unlink(ROOT_PATH . '/public' . $avatarToDelete);
                    }
                    $_SESSION['message'] = "Testimonio eliminado exitosamente.";
                } else {
                    $_SESSION['error'] = "No se pudo eliminar el testimonio.";
                }
            } else {
                 $_SESSION['error'] = "Testimonio no encontrado para eliminar.";
            }
        }
        redirect('admin/testimonials');
    }

    /**
     * Maneja la subida de un archivo de imagen, creando el directorio si no existe.
     */
    private function handleImageUpload(string $fileInputName, ?string $oldImagePath = null): array
    {
        if (!isset($_FILES[$fileInputName]) || $_FILES[$fileInputName]['error'] === UPLOAD_ERR_NO_FILE) {
            return ['success' => true, 'path' => null, 'error' => null];
        }

        $file = $_FILES[$fileInputName];
        if ($file['error'] !== UPLOAD_ERR_OK) {
            return ['success' => false, 'path' => null, 'error' => 'Error durante la subida del archivo. Código: ' . $file['error']];
        }

        $uploadDir = ROOT_PATH . '/public/uploads/testimonials/';

        // --- CLAVE: CREAR LA CARPETA SI NO EXISTE ---
        if (!is_dir($uploadDir)) {
            if (!mkdir($uploadDir, 0775, true)) {
                 return ['success' => false, 'path' => null, 'error' => 'Error del servidor: no se pudo crear el directorio de subida.'];
            }
        }

        $finfo = new finfo(FILEINFO_MIME_TYPE);
        $mimeType = $finfo->file($file['tmp_name']);
        $allowedMimeTypes = ['image/jpeg', 'image/png', 'image/webp'];

        if (!in_array($mimeType, $allowedMimeTypes)) {
            return ['success' => false, 'path' => null, 'error' => 'Formato de archivo no permitido (solo JPG, PNG, WebP).'];
        }

        $fileExtension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        $newFileName = uniqid('testimonial_', true) . '.' . $fileExtension;
        $destPath = $uploadDir . $newFileName;

        if (move_uploaded_file($file['tmp_name'], $destPath)) {
            // Si la subida es exitosa y hay una imagen antigua, bórrala
            if ($oldImagePath && file_exists(ROOT_PATH . '/public' . $oldImagePath)) {
                @unlink(ROOT_PATH . '/public' . $oldImagePath);
            }
            return ['success' => true, 'path' => '/uploads/testimonials/' . $newFileName, 'error' => null];
        }

        return ['success' => false, 'path' => null, 'error' => 'No se pudo mover el archivo subido.'];
    }

    private function renderAdminLayout(string $content, string $pageTitle): void
    {
        ob_start();
        echo $content;
        $content = ob_get_clean();
        require_once ROOT_PATH . '/app/views/admin/layout.php';
    }
}