<?php

require_once ROOT_PATH . '/app/models/Project.php';
require_once ROOT_PATH . '/app/models/Service.php';

class ProjectController
{
    private $db;
    private $project;
    private $service;

    public function __construct($db)
    {
        $this->db = $db;
        $this->project = new Project($this->db);
        $this->service = new Service($this->db);
    }

    /**
     * Muestra la página principal de gestión de proyectos.
     */
    public function index()
    {
        if (!isset($_SESSION['user_id'])) {
            redirect('login');
        }

        $stmt = $this->project->readAll();
        $projects = $stmt->fetchAll(PDO::FETCH_ASSOC);

        ob_start();
        require_once ROOT_PATH . '/app/views/admin/projects/index.php';
        $content = ob_get_clean();
        $this->renderAdminLayout($content, "Gestionar Proyectos");
    }

    /**
     * Muestra el formulario para crear un nuevo proyecto.
     */
    public function create()
    {
        if (!isset($_SESSION['user_id'])) {
            redirect('login');
        }
        $services = $this->getServices();
        $error = null;

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            // 1. Intenta subir la imagen primero
            $uploadResult = $this->handleImageUpload('cover_image');
            
            if ($uploadResult['success']) {
                // 2. Si la subida fue exitosa (o no se subió nada), asigna los datos
                $this->project->id = $_POST['id'] ?? null;
                $this->project->title = $_POST['title'] ?? '';
                // ... (aquí irían todos los demás campos de tu formulario) ...
                $this->project->cta_whatsapp_message = $_POST['cta_whatsapp_message'] ?? '';
                $this->project->cover_image = $uploadResult['path']; // Asigna la nueva ruta o null

                // 3. Intenta crear el proyecto en la base de datos
                if ($this->project->create()) {
                    $_SESSION['message'] = "¡Proyecto creado exitosamente!";
                    redirect('admin/projects');
                } else {
                    $error = "Falló al guardar en la base de datos.";
                    // Si falla la DB, es buena idea borrar la imagen que acabamos de subir
                    if ($uploadResult['path'] && file_exists(ROOT_PATH . '/public' . $uploadResult['path'])) {
                        @unlink(ROOT_PATH . '/public' . $uploadResult['path']);
                    }
                }
            } else {
                // Si la subida de imagen falló, muestra el error específico
                $error = $uploadResult['error'];
            }
        }

        ob_start();
        require_once ROOT_PATH . '/app/views/admin/projects/create.php';
        $content = ob_get_clean();
        $this->renderAdminLayout($content, "Crear Nuevo Proyecto");
    }

    /**
     * Muestra el formulario para editar un proyecto existente.
     */
   public function edit()
{
    if (!isset($_SESSION['user_id'])) { redirect('login'); }

    $this->project->id = $_GET['id'] ?? null;
    if (!$this->project->id || !$this->project->readOne()) {
        $_SESSION['error'] = "Proyecto no encontrado.";
        redirect('admin/projects');
    }
    
    // Decodifica la galería de JSON a un array de PHP al cargar
    if (!is_array($this->project->gallery)) {
        $this->project->gallery = json_decode($this->project->gallery, true) ?? [];
    }

    $services = $this->getServices();
    $error = null;

    if ($_SERVER['REQUEST_METHOD'] === 'POST') {
        // --- LÓGICA DE LA GALERÍA ---

        // 1. MANEJAR IMÁGENES A ELIMINAR
        $imagesToDelete = $_POST['images_to_delete'] ?? [];
        if (!empty($imagesToDelete)) {
            // Filtra la galería actual, quedándote solo con las imágenes que NO están marcadas para eliminar
            $this->project->gallery = array_filter($this->project->gallery, function($imagePath) use ($imagesToDelete) {
                return !in_array($imagePath, $imagesToDelete);
            });

            // Borra los archivos físicos del servidor
            foreach ($imagesToDelete as $imagePath) {
                $fullPath = ROOT_PATH . '/public' . $imagePath;
                if (file_exists($fullPath)) {
                    @unlink($fullPath);
                }
            }
        }

        // 2. MANEJAR NUEVAS IMÁGENES SUBIDAS
        $galleryUploadResult = $this->handleGalleryUpload('gallery');
        if ($galleryUploadResult['success']) {
            // Combina la galería existente con las nuevas imágenes subidas
            $this->project->gallery = array_merge($this->project->gallery, $galleryUploadResult['paths']);
        } else {
            // Si la subida de la galería falla, muestra un error
            $error = $galleryUploadResult['error'];
        }
        
        // --- FIN LÓGICA GALERÍA ---

        // Ahora, manejamos la imagen de portada como antes
        $uploadResult = $this->handleImageUpload('cover_image', $this->project->cover_image);

        if ($uploadResult['success'] && $error === null) {
            // Asigna todos los datos del formulario
            $this->project->title = $_POST['title'] ?? '';
            // ... (asigna todos los demás campos de texto) ...
            $this->project->cta_whatsapp_message = $_POST['cta_whatsapp_message'] ?? '';
            
            if ($uploadResult['path'] !== null) {
                $this->project->cover_image = $uploadResult['path'];
            }

            // Antes de actualizar, asegúrate de que la galería sea un string JSON
            $this->project->gallery = json_encode(array_values($this->project->gallery)); // Re-indexar y codificar

            // Intenta actualizar la base de datos
            if ($this->project->update()) {
                $_SESSION['message'] = "¡Proyecto actualizado exitosamente!";
                redirect('admin/projects');
            } else {
                $error = "Falló al actualizar la base de datos.";
            }
        } else {
            // Si la subida de la portada falló, también lo mostramos
            $error = $error ?? $uploadResult['error'];
        }
    }

    ob_start();
    require_once ROOT_PATH . '/app/views/admin/projects/edit.php';
    $content = ob_get_clean();
    $this->renderAdminLayout($content, "Editar Proyecto");
}
private function handleGalleryUpload(string $fileInputName): array
{
    // Verifica si se subieron archivos
    if (!isset($_FILES[$fileInputName]) || empty($_FILES[$fileInputName]['name'][0])) {
        return ['success' => true, 'paths' => [], 'error' => null];
    }

    $files = $_FILES[$fileInputName];
    $uploadedPaths = [];
    $errors = [];
    $uploadDir = ROOT_PATH . '/public/uploads/projects/';

    // Asegúrate de que el directorio exista
    if (!is_dir($uploadDir)) {
        if (!mkdir($uploadDir, 0775, true)) {
            return ['success' => false, 'paths' => [], 'error' => 'Error: No se pudo crear el directorio de la galería.'];
        }
    }
    
    // Itera sobre cada archivo subido
    foreach ($files['name'] as $key => $name) {
        if ($files['error'][$key] === UPLOAD_ERR_OK) {
            $tmp_name = $files['tmp_name'][$key];
            
            // Validaciones (puedes añadir más si quieres)
            $finfo = new finfo(FILEINFO_MIME_TYPE);
            $mimeType = $finfo->file($tmp_name);
            $allowedMimeTypes = ['image/jpeg', 'image/png', 'image/webp'];
            if (!in_array($mimeType, $allowedMimeTypes)) {
                $errors[] = "Archivo '$name' ignorado: formato no permitido.";
                continue; // Salta al siguiente archivo
            }

            // Proceso de guardado
            $fileExtension = strtolower(pathinfo($name, PATHINFO_EXTENSION));
            $newFileName = uniqid('gallery_', true) . '.' . $fileExtension;
            $destPath = $uploadDir . $newFileName;

            if (move_uploaded_file($tmp_name, $destPath)) {
                $uploadedPaths[] = '/uploads/projects/' . $newFileName;
            } else {
                $errors[] = "No se pudo mover el archivo '$name'.";
            }
        }
    }

    if (!empty($errors)) {
        // Devuelve éxito parcial si algunos archivos subieron y otros no
        return ['success' => false, 'paths' => $uploadedPaths, 'error' => implode(' ', $errors)];
    }

    return ['success' => true, 'paths' => $uploadedPaths, 'error' => null];
}
    /**
     * Procesa la eliminación de un proyecto.
     */
    public function delete()
    {
        if (!isset($_SESSION['user_id'])) { redirect('login'); }
        
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $this->project->id = $_POST['id'] ?? null;

            // Primero, obtenemos la ruta de la imagen antes de borrar el registro
            if ($this->project->readOne()) {
                $imageToDelete = $this->project->cover_image;

                // Ahora, intentamos borrar de la base de datos
                if ($this->project->delete()) {
                    // Si se borró de la DB, borramos el archivo de imagen
                    if ($imageToDelete && file_exists(ROOT_PATH . '/public' . $imageToDelete)) {
                        @unlink(ROOT_PATH . '/public' . $imageToDelete);
                    }
                    $_SESSION['message'] = "Proyecto eliminado exitosamente.";
                } else {
                    $_SESSION['error'] = "No se pudo eliminar el proyecto.";
                }
            } else {
                 $_SESSION['error'] = "Proyecto no encontrado para eliminar.";
            }
        }
        redirect('admin/projects');
    }

    /**
     * Renderiza el layout principal del panel de administración.
     */
    private function renderAdminLayout($content, $pageTitle)
    {
        ob_start();
        echo $content;
        $content = ob_get_clean();
        require_once ROOT_PATH . '/app/views/admin/layout.php';
    }

    /**
     * Obtiene todos los servicios disponibles.
     */
    private function getServices()
    {
        $stmt = $this->service->readAll();
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    /**
     * Maneja la subida de un archivo de imagen, creando el directorio si no existe.
     */
    private function handleImageUpload(string $fileInputName, ?string $oldImagePath = null): array
    {
        if (!isset($_FILES[$fileInputName]) || $_FILES[$fileInputName]['error'] === UPLOAD_ERR_NO_FILE) {
            return ['success' => true, 'path' => null, 'error' => null];
        }

        $file = $_FILES[$fileInputName];

        if ($file['error'] !== UPLOAD_ERR_OK) {
            $phpUploadErrors = [
                UPLOAD_ERR_INI_SIZE   => 'El archivo excede el tamaño máximo permitido por el servidor.',
                UPLOAD_ERR_FORM_SIZE  => 'El archivo excede el tamaño máximo permitido en el formulario.',
                UPLOAD_ERR_PARTIAL    => 'El archivo fue solo parcialmente subido.',
                UPLOAD_ERR_CANT_WRITE => 'No se pudo escribir el archivo en el disco.',
                UPLOAD_ERR_EXTENSION  => 'Una extensión de PHP detuvo la subida del archivo.',
            ];
            return ['success' => false, 'path' => null, 'error' => $phpUploadErrors[$file['error']] ?? 'Error desconocido al subir el archivo.'];
        }

        $uploadDir = ROOT_PATH . '/public/uploads/projects/';

        if (!is_dir($uploadDir)) {
            if (!mkdir($uploadDir, 0775, true)) {
                return ['success' => false, 'path' => null, 'error' => 'Error del servidor: no se pudo crear el directorio de subida. Verifica los permisos.'];
            }
        }

        if (!is_writable($uploadDir)) {
             return ['success' => false, 'path' => null, 'error' => 'Error del servidor: el directorio de subida no tiene permisos de escritura.'];
        }

        $finfo = new finfo(FILEINFO_MIME_TYPE);
        $mimeType = $finfo->file($file['tmp_name']);
        $allowedMimeTypes = ['image/jpeg', 'image/png', 'image/webp'];

        if (!in_array($mimeType, $allowedMimeTypes)) {
            return ['success' => false, 'path' => null, 'error' => 'Formato de archivo no permitido. Sube solo imágenes JPG, PNG o WebP.'];
        }

        $fileExtension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
        $newFileName = uniqid('proj_', true) . '.' . $fileExtension;
        $destPath = $uploadDir . $newFileName;

        if (!move_uploaded_file($file['tmp_name'], $destPath)) {
             return ['success' => false, 'path' => null, 'error' => 'No se pudo mover el archivo subido a su destino.'];
        }

        if ($oldImagePath && file_exists(ROOT_PATH . '/public' . $oldImagePath)) {
            @unlink(ROOT_PATH . '/public' . $oldImagePath);
        }

        return ['success' => true, 'path' => '/uploads/projects/' . $newFileName, 'error' => null];
    }
}