<?php

require_once ROOT_PATH . '/app/models/Product.php';

/**
 * Class ProductController
 *
 * Gestiona la lógica CRUD para los productos.
 */
class ProductController
{
    private PDO $db;
    private Product $product;

    public function __construct(PDO $db)
    {
        $this->db = $db;
        $this->product = new Product($this->db);
    }

    /**
     * Muestra la lista de todos los productos.
     */
    public function index(): void
    {
        if (!isset($_SESSION['user_id'])) {
            redirect('login');
        }

        $stmt = $this->product->readAll();
        $products = $stmt->fetchAll(PDO::FETCH_ASSOC);

        ob_start();
        require_once ROOT_PATH . '/app/views/admin/products/index.php';
        $content = ob_get_clean();
        $this->renderAdminLayout($content, "Gestionar Productos");
    }

    /**
     * Muestra el formulario para crear un nuevo producto o procesa su creación.
     */
    public function create(): void
    {
        if (!isset($_SESSION['user_id'])) {
            redirect('login');
        }

        $error = null;
        $productData = []; // Para repoblar el formulario en caso de error

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $productData = $_POST; // Guarda los datos enviados por el usuario

            $this->product->id = $_POST['id'] ?? '';
            $this->product->title = $_POST['title'] ?? '';
            $this->product->price = $_POST['price'] ?? '';
            $this->product->short_description = $_POST['short_description'] ?? '';
            $this->product->ideal_for = $_POST['ideal_for'] ?? '';
            // Limpia saltos de línea vacíos antes de procesar
            $this->product->included_modules = isset($_POST['included_modules']) ? array_filter(array_map('trim', explode("\n", $_POST['included_modules']))) : [];
            $this->product->cta_text = $_POST['cta_text'] ?? '';
            $this->product->cta_whatsapp_message = $_POST['cta_whatsapp_message'] ?? '';

            if ($this->product->create()) {
                $_SESSION['message'] = "¡Producto creado exitosamente!";
                redirect('admin/products');
            } else {
                $error = "Falló al crear el producto. Revisa que el ID no esté repetido.";
            }
        }

        ob_start();
        // Pasamos $productData a la vista para que pueda repoblar los campos
        require_once ROOT_PATH . '/app/views/admin/products/create.php';
        $content = ob_get_clean();
        $this->renderAdminLayout($content, "Crear Nuevo Producto");
    }

    /**
     * Muestra el formulario para editar un producto o procesa su actualización.
     */
    public function edit(): void
    {
        if (!isset($_SESSION['user_id'])) {
            redirect('login');
        }

        $this->product->id = $_GET['id'] ?? '';
        if (!$this->product->readOne()) {
            $_SESSION['error'] = "Producto no encontrado.";
            redirect('admin/products');
        }

        $error = null;

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            // El ID no se actualiza, pero se necesita para la consulta
            $this->product->id = $_POST['id'] ?? $this->product->id;
            $this->product->title = $_POST['title'] ?? '';
            $this->product->price = $_POST['price'] ?? '';
            $this->product->short_description = $_POST['short_description'] ?? '';
            $this->product->ideal_for = $_POST['ideal_for'] ?? '';
            $this->product->included_modules = isset($_POST['included_modules']) ? array_filter(array_map('trim', explode("\n", $_POST['included_modules']))) : [];
            $this->product->cta_text = $_POST['cta_text'] ?? '';
            $this->product->cta_whatsapp_message = $_POST['cta_whatsapp_message'] ?? '';

            if ($this->product->update()) {
                $_SESSION['message'] = "¡Producto actualizado exitosamente!";
                redirect('admin/products');
            } else {
                $error = "Falló al actualizar el producto.";
            }
        }

        ob_start();
        require_once ROOT_PATH . '/app/views/admin/products/edit.php';
        $content = ob_get_clean();
        $this->renderAdminLayout($content, "Editar Producto");
    }

    /**
     * Procesa la eliminación de un producto de forma segura (vía POST).
     */
    public function delete(): void
    {
        if (!isset($_SESSION['user_id'])) {
            redirect('login');
        }
        
        // La eliminación segura se hace por POST
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $this->product->id = $_POST['id'] ?? '';
            if ($this->product->delete()) {
                $_SESSION['message'] = "¡Producto eliminado exitosamente!";
            } else {
                $_SESSION['error'] = "Falló al eliminar el producto.";
            }
        }
        redirect('admin/products');
    }

    private function renderAdminLayout(string $content, string $pageTitle): void
    {
        ob_start();
        echo $content;
        $content = ob_get_clean();
        require_once ROOT_PATH . '/app/views/admin/layout.php';
    }
}