<?php

require_once ROOT_PATH . '/app/models/Page.php';

/**
 * Class PageController
 *
 * Gestiona la lógica para listar y editar las páginas del sitio.
 */
class PageController
{
    private PDO $db;
    private Page $page;

    /**
     * @param PDO $db Conexión a la base de datos.
     */
    public function __construct(PDO $db)
    {
        $this->db = $db;
        $this->page = new Page($this->db);
    }

    /**
     * Muestra la lista de todas las páginas gestionables.
     */
    public function index(): void
    {
        if (!isset($_SESSION['user_id'])) {
            // Usa la función redirect() para la URL correcta
            redirect('login');
        }

        $stmt = $this->page->readAll();
        $pages = $stmt->fetchAll(PDO::FETCH_ASSOC);

        ob_start();
        require_once ROOT_PATH . '/app/views/admin/pages/index.php';
        $content = ob_get_clean();
        $this->renderAdminLayout($content, "Gestionar Páginas");
    }

    /**
     * Muestra el formulario para editar una página o procesa los datos del formulario.
     */
    public function edit(): void
    {
        if (!isset($_SESSION['user_id'])) {
            // Usa la función redirect() para la URL correcta
            redirect('login');
        }

        $this->page->page_key = $_GET['page_key'] ?? '';

        if (!$this->page->readOne()) {
            $_SESSION['error'] = "Página no encontrada.";
            // Usa la función redirect() para la URL correcta
            redirect('admin/pages');
        }

        $error = null;

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            // Asigna los valores comunes
            $this->page->seo_title = $_POST['seo_title'] ?? '';
            $this->page->seo_description = $_POST['seo_description'] ?? '';

            // Asigna valores específicos según la página que se está editando
            if ($this->page->page_key === 'home') {
                $this->page->hero_headline = $_POST['hero_headline'] ?? null;
                $this->page->hero_subheadline = $_POST['hero_subheadline'] ?? null;
                $this->page->hero_cta_primary_text = $_POST['hero_cta_primary_text'] ?? null;
                $this->page->hero_cta_primary_href = $_POST['hero_cta_primary_href'] ?? null;
                $this->page->hero_cta_secondary_text = $_POST['hero_cta_secondary_text'] ?? null;
                $this->page->hero_cta_secondary_href = $_POST['hero_cta_secondary_href'] ?? null;
                $this->page->stats = isset($_POST['stats']) ? json_decode($_POST['stats'], true) : [];
            } elseif ($this->page->page_key === 'about') {
                $this->page->mission = $_POST['mission'] ?? null;
                $this->page->vision = $_POST['vision'] ?? null;
                $this->page->timeline = isset($_POST['timeline']) ? json_decode($_POST['timeline'], true) : [];
            }
            
            if ($this->page->update()) {
                $_SESSION['message'] = "¡Página '" . htmlspecialchars($this->page->page_key) . "' actualizada exitosamente!";
                // Usa la función redirect() para la URL correcta
                redirect('admin/pages');
            } else {
                $error = "Error al actualizar la página.";
            }
        }

        ob_start();
        require_once ROOT_PATH . '/app/views/admin/pages/edit.php';
        $content = ob_get_clean();
        $this->renderAdminLayout($content, "Editar Página: " . ucfirst($this->page->page_key));
    }
    
    /**
     * Renderiza el layout principal del panel de administración.
     * @param string $content El HTML de la vista a renderizar.
     * @param string $pageTitle El título de la página.
     */
    private function renderAdminLayout(string $content, string $pageTitle): void
    {
        ob_start();
        echo $content;
        $content = ob_get_clean();
        require_once ROOT_PATH . '/app/views/admin/layout.php';
    }
}