<?php

// Asegúrate de que la ruta al modelo sea correcta según tu estructura de archivos.
require_once ROOT_PATH . '/app/models/MethodologyStep.php';

/**
 * Class MethodologyStepController
 *
 * Gestiona la lógica CRUD para los Pasos de la Metodología (Methodology Steps).
 */
class MethodologyStepController
{
    private PDO $db;
    private MethodologyStep $methodologyStep;

    public function __construct(PDO $db)
    {
        $this->db = $db;
        $this->methodologyStep = new MethodologyStep($this->db);
    }

    /**
     * Muestra una lista de todos los pasos de la metodología.
     */
    public function index(): void
    {
        // Proteger la página: solo para usuarios autenticados
        if (!isset($_SESSION['user_id'])) {
            redirect('login');
        }

        // Obtener todos los registros
        $steps = $this->methodologyStep->readAll()->fetchAll(PDO::FETCH_ASSOC);

        // Renderizar la vista
        ob_start();
        // Asegúrate de que esta ruta de vista exista
        require_once ROOT_PATH . '/app/views/admin/methodology_steps/index.php';
        $content = ob_get_clean();
        $this->renderAdminLayout($content, "Gestionar Pasos de Metodología");
    }

    /**
     * Muestra el formulario para crear un nuevo paso y procesa su envío.
     */
    public function create(): void
    {
        if (!isset($_SESSION['user_id'])) {
            redirect('login');
        }

        $error = null;
        $stepData = []; // Para repoblar el formulario en caso de error

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            // Guardar datos del POST para repoblar si hay un error
            $stepData = $_POST;
            
            // Asignar propiedades al objeto del modelo
            $this->methodologyStep->title = $_POST['title'] ?? '';
            $this->methodologyStep->description = $_POST['description'] ?? '';
            $this->methodologyStep->icon = $_POST['icon'] ?? '';
            $this->methodologyStep->display_order = !empty($_POST['display_order']) ? (int)$_POST['display_order'] : 0;

            if ($this->methodologyStep->create()) {
                $_SESSION['message'] = "¡Paso creado exitosamente!";
                redirect('admin/methodology_steps');
            } else {
                $error = "Falló al crear el paso de la metodología.";
            }
        }

        ob_start();
        // Asegúrate de que esta ruta de vista exista
        require_once ROOT_PATH . '/app/views/admin/methodology_steps/create.php';
        $content = ob_get_clean();
        $this->renderAdminLayout($content, "Crear Nuevo Paso");
    }

    /**
     * Muestra el formulario para editar un paso existente y procesa su actualización.
     */
    public function edit(): void
    {
        if (!isset($_SESSION['user_id'])) {
            redirect('login');
        }

        $error = null;
        // Obtener ID desde la URL
        $this->methodologyStep->id = isset($_GET['id']) ? (int)$_GET['id'] : 0;

        // Verificar si el registro existe
        if (!$this->methodologyStep->readOne()) {
            $_SESSION['error'] = "Paso de metodología no encontrado.";
            redirect('admin/methodology_steps');
        }

        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            // Asignar propiedades para la actualización
            $this->methodologyStep->id = (int)$_POST['id'];
            $this->methodologyStep->title = $_POST['title'] ?? '';
            $this->methodologyStep->description = $_POST['description'] ?? '';
            $this->methodologyStep->icon = $_POST['icon'] ?? '';
            $this->methodologyStep->display_order = !empty($_POST['display_order']) ? (int)$_POST['display_order'] : 0;
            
            if ($this->methodologyStep->update()) {
                $_SESSION['message'] = "¡Paso actualizado exitosamente!";
                redirect('admin/methodology_steps');
            } else {
                $error = "Falló al actualizar el paso.";
            }
        }

        ob_start();
        // Asegúrate de que esta ruta de vista exista
        require_once ROOT_PATH . '/app/views/admin/methodology_steps/edit.php';
        $content = ob_get_clean();
        $this->renderAdminLayout($content, "Editar Paso");
    }

    /**
     * Procesa la eliminación de un paso.
     */
    public function delete(): void
    {
        if (!isset($_SESSION['user_id'])) {
            redirect('login');
        }

        // La eliminación segura se hace por POST para evitar borrado accidental por URL
        if ($_SERVER['REQUEST_METHOD'] === 'POST') {
            $this->methodologyStep->id = isset($_POST['id']) ? (int)$_POST['id'] : 0;
            if ($this->methodologyStep->delete()) {
                $_SESSION['message'] = "¡Paso eliminado exitosamente!";
            } else {
                $_SESSION['error'] = "Falló al eliminar el paso.";
            }
        }
        redirect('admin/methodology_steps');
    }

    /**
     * Renderiza el contenido principal dentro del layout del panel de administración.
     * @param string $content El HTML del contenido principal.
     * @param string $pageTitle El título de la página.
     */
    private function renderAdminLayout(string $content, string $pageTitle): void
    {
        ob_start();
        echo $content;
        $pageContent = ob_get_clean(); // Renombrado para evitar confusión
        require_once ROOT_PATH . '/app/views/admin/layout.php';
    }
}
